/* BG+GIF Overlay: image2cpp mono bitmap (220x176) + 128x128 RGB565 GIF
 * TFT: ILI9225 (TFT_22_ILI9225)
 */

#include <SPI.h>
#include <TFT_22_ILI9225.h>
#include <pgmspace.h>

// ---------- Pins ----------
#define TFT_CS    D8
#define TFT_RS    D1   // D/C
#define TFT_RST   D2
#define TFT_LED   D4
#define TFT_BRIGHTNESS 200

TFT_22_ILI9225 tft(TFT_RST, TFT_RS, TFT_CS, TFT_LED, TFT_BRIGHTNESS);

// ---------- Display ----------
const int T_W = 220, T_H = 176;

// ---------- Background (image2cpp) ----------
#define BMP_W 220
#define BMP_H 176
extern const unsigned char epd_bitmap_img[] PROGMEM;  // from image2cpp
// Colors for 1-bit bitmap:
#define BMP_COLOR_FG  COLOR_WHITE
#define BMP_COLOR_BG  COLOR_BLACK   // if you want "transparent bg", set DRAW_BG=false
#define DRAW_BG       true          // true: draw background pixels too; false: only FG

// ---------- GIF (RGB565) ----------
#define GIF_W 128
#define GIF_H 128
#define FRAME_PIXELS (GIF_W * GIF_H)
#define FRAME_DELAY_MS   30        // smoother than 60 (try 20–30)
#define DO_CHANNEL_SWAP  false     // blue tint fix; toggle if colors wrong

// try fast line push; set 0 if your lib lacks setWindow/_writeData16
#define USE_FAST_LINE_PUSH 1

// Your GIF frames (uint16_t RGB565):
#include "animation.h"   // const uint16_t animation[][16384] PROGMEM;
#ifndef ANIM_FRAME_COUNT
  const size_t GIF_FRAME_COUNT = sizeof(animation) / sizeof(animation[0]);
#else
  const size_t GIF_FRAME_COUNT = ANIM_FRAME_COUNT;
#endif

// ---------- Helpers ----------
static inline uint16_t rgb_to_bgr_565(uint16_t c) {
  return (uint16_t)(((c & 0x001F) << 11) | (c & 0x07E0) | ((c & 0xF800) >> 11));
}
static inline uint16_t gif_px(size_t f, uint32_t i) {
  uint16_t c = pgm_read_word(&animation[f][i]);
  if (DO_CHANNEL_SWAP) c = rgb_to_bgr_565(c);
  return c;
}

// Draw 1-bit mono bitmap generated by image2cpp (packed MSB->LSB per byte)
void drawMonoBitmap(int16_t x, int16_t y, const uint8_t *bitmap,
                    int16_t w, int16_t h,
                    uint16_t colorFG, uint16_t colorBG,
                    bool drawBG) {
  // each row has w pixels => (w+7)/8 bytes
  int16_t byteWidth = (w + 7) / 8;
  for (int16_t j = 0; j < h; j++) {
    for (int16_t i = 0; i < w; i++) {
      uint8_t b = pgm_read_byte(bitmap + j * byteWidth + (i >> 3));
      bool bitOn = b & (0x80 >> (i & 7));
      if (bitOn) {
        tft.drawPixel(x + i, y + j, colorFG);
      } else if (drawBG) {
        tft.drawPixel(x + i, y + j, colorBG);
      }
    }
  }
}

// ---------- GIF draw (overlay, no clear) ----------
#if USE_FAST_LINE_PUSH
void drawGIFFrameFast(size_t frameIndex) {
  const int x0 = (T_W - GIF_W) / 2;
  const int y0 = (T_H - GIF_H) / 2;
  static uint16_t line[GIF_W];

  for (int y = 0; y < GIF_H; y++) {
    const uint32_t base = (uint32_t)y * GIF_W;
    for (int x = 0; x < GIF_W; x++) line[x] = gif_px(frameIndex, base + x);

    // Try internal window/push (many forks expose these; if not, fallback below)
    #if defined(ILI9225_REG) || defined(_ILI9225H_) || defined(ILI9225_DRIVER)
      tft.setWindow(x0, y0 + y, x0 + GIF_W - 1, y0 + y);
      tft._writeData16(line, GIF_W);
    #else
      for (int x = 0; x < GIF_W; x++) tft.drawPixel(x0 + x, y0 + y, line[x]);
    #endif
  }
}
#endif

void drawGIFFrameSafe(size_t frameIndex) {
  const int x0 = (T_W - GIF_W) / 2;
  const int y0 = (T_H - GIF_H) / 2;

  for (int y = 0; y < GIF_H; y++) {
    const uint32_t base = (uint32_t)y * GIF_W;
    for (int x = 0; x < GIF_W; x++) {
      tft.drawPixel(x0 + x, y0 + y, gif_px(frameIndex, base + x));
    }
  }
}

// ---------- Setup / Loop ----------
void setup() {
  tft.begin();
  tft.setOrientation(1); // landscape 220x176

  // Draw full-screen background once
  drawMonoBitmap(0, 0, epd_bitmap_img, BMP_W, BMP_H, BMP_COLOR_FG, BMP_COLOR_BG, DRAW_BG);
}

void loop() {
  for (size_t i = 0; i < GIF_FRAME_COUNT; i++) {
    #if USE_FAST_LINE_PUSH
      drawGIFFrameFast(i);
    #else
      drawGIFFrameSafe(i);
    #endif
    // zyada smooth feel ke liye delay ko 20–30ms pe tune karo
    delay(FRAME_DELAY_MS);
    yield();
  }
}
